// SPDX-FileCopyrightText: 2024 g10 code Gmbh
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "firsttimedialog.h"
#include "config.h"
#include "gpgolweb_version.h"
#include "rootcagenerator/controller.h"
#include "ui_confpageinstalladdin.h"
#include "ui_confpageproxyoptions.h"
#include "ui_confpagetlscertificate.h"
#include "ui_confpagewelcome.h"
#include "ui_firsttimedialog.h"
#include "websocketclient.h"

#include <QCheckBox>
#include <QClipboard>
#include <QCloseEvent>
#include <QDesktopServices>
#include <QFile>
#include <QMargins>
#include <QSaveFile>
#include <QStandardPaths>
#include <QStatusBar>
#include <QStyle>
#include <QTemporaryDir>
#include <QToolBar>

#include <Libkleo/Compliance>

#include <KColorScheme>
#include <KIO/OpenFileManagerWindowJob>
#include <KTitleWidget>

using namespace Qt::StringLiterals;

FirstTimeDialog::FirstTimeDialog(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::FirstTimeDialog)
    , confPageWelcome(new Ui::ConfPageWelcome)
    , confPageInstallAddin(new Ui::ConfPageInstallAddin)
    , confPageProxyOptions(new Ui::ConfPageProxyOptions)
    , confPageTLSCertificate(new Ui::ConfPageTLSCertificate)
    , m_systemTrayIcon(QIcon::fromTheme(u"com.gnupg.gpgolweb"_s))
{
    ui->setupUi(this);
    confPageWelcome->setupUi(ui->confPageWelcome);
    ui->confPageWelcome->setProperty("title", i18nc("@title", "Welcome to GpgOL/Web"));
    confPageProxyOptions->setupUi(ui->confPageProxyOptions);
    ui->confPageProxyOptions->setProperty("title", i18nc("@title", "Configure Proxy and Optional Features"));
    confPageInstallAddin->setupUi(ui->confPageInstallAddin);
    ui->confPageInstallAddin->setProperty("title", i18nc("@title", "Install Outlook Add-In"));
    confPageTLSCertificate->setupUi(ui->confPageTLSCertificate);
    ui->confPageTLSCertificate->setProperty("title", i18nc("@title", "Setting Up TLS Certificate for Local Proxy"));

    if (ui->stack->indexOf(ui->confPageWelcome) != ConfPageWelcome) {
        qFatal("Welcome page misplaced");
    }
    if (ui->stack->indexOf(ui->confPageTLSCertificate) != ConfPageTLSCertificate) {
        qFatal("Tls certification page misplaced");
    }
    if (ui->stack->indexOf(ui->confPageProxyOptions) != ConfPageProxyOptions) {
        qFatal("Proxy options page misplaced");
    }
    if (ui->stack->indexOf(ui->confPageInstallAddin) != ManifestPage) {
        qFatal("Manifest install page misplaced");
    }

    confPageProxyOptions->reencryptOption->setChecked(Config::self()->reencrypt());
    connect(confPageProxyOptions->reencryptOption, &QCheckBox::stateChanged, this, [](int state) {
        Config::self()->setReencrypt(state == Qt::Checked);
        Config::self()->save();
    });

    auto margins = confPageProxyOptions->remoteServerLayout->contentsMargins();
    margins.setLeft(margins.left() + style()->pixelMetric(QStyle::PM_RadioButtonLabelSpacing) + style()->pixelMetric(QStyle::PM_ExclusiveIndicatorWidth));
    confPageProxyOptions->remoteServerLayout->setContentsMargins(margins);

    m_systemTrayIcon.setMainWindow(this);
    m_systemTrayIcon.show();

    m_backAction = new QAction(this);
    connect(m_backAction, &QAction::triggered, this, [this]() {
        if (Controller::certificateAlreadyGenerated() || !Config::self()->isLocalServer()) {
            ui->stack->setCurrentIndex(ui->stack->currentIndex() > 1 ? ConfPageProxyOptions : ConfPageWelcome);
        } else {
            ui->stack->setCurrentIndex(ui->stack->currentIndex() == ConfPageProxyOptions ? ConfPageWelcome : ConfPageProxyOptions);
        }
    });
    connect(confPageTLSCertificate->backButton, &QAbstractButton::clicked, m_backAction, &QAction::trigger);
    connect(confPageInstallAddin->backButton, &QAbstractButton::clicked, m_backAction, &QAction::trigger);
    connect(confPageProxyOptions->backButton, &QAbstractButton::clicked, m_backAction, &QAction::trigger);
    auto toolbar = new QToolBar(this);
    toolbar->setMovable(false);
    auto titleWidget = new KTitleWidget(this);
    toolbar->addWidget(titleWidget);
    addToolBar(Qt::TopToolBarArea, toolbar);

    titleWidget->setText(ui->stack->currentWidget()->property("title").toString());
    connect(ui->stack, &QStackedWidget::currentChanged, this, [titleWidget, this]() {
        titleWidget->setText(ui->stack->currentWidget()->property("title").toString());
    });

    QPixmap logo = QIcon::fromTheme(u"com.gnupg.gpgolweb"_s).pixmap(64, 64);
    confPageWelcome->logo->setPixmap(logo);
    confPageWelcome->titleWelcome->setText(i18nc("@info", "GpgOL/Web %1", QString::fromLocal8Bit(GPGOLWEB_VERSION_STRING)));

    auto statusBar = new QStatusBar(this);

    confPageInstallAddin->showOnStartup->setChecked(Config::self()->showLauncher());
    connect(confPageInstallAddin->showOnStartup, &QCheckBox::toggled, this, [](bool checked) {
        Config::self()->setShowLauncher(checked);
        Config::self()->save();
    });

    m_status = new QLabel;
    statusBar->addPermanentWidget(m_status);

    auto version = new QLabel(i18nc("@info", "Version: %1", QString::fromLocal8Bit(GPGOLWEB_VERSION_STRING)));
    statusBar->addPermanentWidget(version);

    if (Kleo::DeVSCompliance::isActive()) {
        auto statusLbl = std::make_unique<QLabel>(Kleo::DeVSCompliance::name());
        {
            auto statusPalette = qApp->palette();
            KColorScheme::adjustForeground(statusPalette,
                                           Kleo::DeVSCompliance::isCompliant() ? KColorScheme::NormalText : KColorScheme::NegativeText,
                                           statusLbl->foregroundRole(),
                                           KColorScheme::View);
            statusLbl->setAutoFillBackground(true);
            KColorScheme::adjustBackground(statusPalette,
                                           Kleo::DeVSCompliance::isCompliant() ? KColorScheme::PositiveBackground : KColorScheme::NegativeBackground,
                                           QPalette::Window,
                                           KColorScheme::View);
            statusLbl->setPalette(statusPalette);
        }
        statusBar->addPermanentWidget(statusLbl.release());
    }

    setStatusBar(statusBar);

    connect(confPageProxyOptions->continueButton, &QPushButton::clicked, this, &FirstTimeDialog::slotSetup);
    connect(confPageWelcome->configureButton, &QPushButton::clicked, this, [this]() {
        ui->stack->setCurrentIndex(ConfPageProxyOptions);
    });

    connect(confPageTLSCertificate->continueButton, &QPushButton::clicked, this, &FirstTimeDialog::slotSetup);
    confPageTLSCertificate->continueButton->setEnabled(false);
    confPageTLSCertificate->installButton->setVisible(false);
    confPageTLSCertificate->label->setVisible(false);
    connect(confPageTLSCertificate->installButton, &QPushButton::clicked, this, [this]() {
        if (m_controller) {
            m_controller->install();
        }
    });

    confPageInstallAddin->manifestPath->setText(QLatin1StringView(DATAROUTDIR) + u"/gpgol/manifest.xml"_s);

    connect(confPageInstallAddin->openOutlookButton, &QPushButton::clicked, this, []() {
        QDesktopServices::openUrl(QUrl(u"https://outlook.office.com/mail/jsmvvmdeeplink/?path=/options/manageapps&bO=4"_s));
    });
    connect(confPageInstallAddin->testPageButton, &QPushButton::clicked, this, [this]() {
        QDesktopServices::openUrl(QUrl(u"https://"_s + serverDomain() + u"/test"_s));
    });
    connect(confPageInstallAddin->minimizeButton, &QPushButton::clicked, this, &QWidget::hide);

    connect(confPageInstallAddin->manifestPathCopy, &QPushButton::clicked, this, [this]() {
        QGuiApplication::clipboard()->setText(confPageInstallAddin->manifestPath->text());
    });

    connect(confPageInstallAddin->manifestPathOpenFolder, &QPushButton::clicked, this, [this]() {
        auto job = new KIO::OpenFileManagerWindowJob();
        job->setHighlightUrls({QUrl::fromUserInput(confPageInstallAddin->manifestPath->text())});
        if (!qEnvironmentVariableIsEmpty("XDG_ACTIVATION_TOKEN")) {
            job->setStartupId(qgetenv("XDG_ACTIVATION_TOKEN"));
        }
        job->start();
    });

    confPageProxyOptions->remoteLabel->setEnabled(!Config::self()->isLocalServer());
    confPageProxyOptions->remoteServer->setEnabled(!Config::self()->isLocalServer());
    confPageProxyOptions->remoteServer->setText(Config::self()->remoteAddress().toString());
    confPageProxyOptions->remoteOption->setChecked(!Config::self()->isLocalServer());

    connect(confPageProxyOptions->remoteOption, &QRadioButton::toggled, this, [this](bool checked) {
        Config::self()->setIsLocalServer(!checked);
        Config::self()->save();

        confPageProxyOptions->remoteLabel->setEnabled(!Config::self()->isLocalServer());
        confPageProxyOptions->remoteServer->setEnabled(!Config::self()->isLocalServer());
    });

    connect(confPageProxyOptions->remoteServer, &QLineEdit::textChanged, this, [this]() {
        Config::self()->setRemoteAddress(QUrl::fromUserInput(confPageProxyOptions->remoteServer->text()));
        Config::self()->save();
    });

    if (Controller::certificateAlreadyGenerated() || !Config::self()->isLocalServer()) {
        ui->stack->setCurrentIndex(ConfPageWelcome);
        if (Controller::certificateAlreadyGenerated() && Config::self()->isLocalServer()) {
            startLocalServer();
        }
        startWebsocketClient();
    } else {
        ui->stack->setCurrentIndex(ConfPageProxyOptions);
    }

    connect(&m_serverProcess, &QProcess::readyReadStandardError, this, [this]() {
        qWarning().noquote() << m_serverProcess.readAllStandardError();
    });

    connect(&m_serverProcess, &QProcess::readyReadStandardOutput, this, [this]() {
        qWarning().noquote() << m_serverProcess.readAllStandardOutput();
    });
}

FirstTimeDialog::~FirstTimeDialog() = default;

void FirstTimeDialog::slotStateChanged(const QString &stateDisplay)
{
    m_status->setText(stateDisplay);
}

void FirstTimeDialog::closeEvent(QCloseEvent *e)
{
    e->ignore();
    hide();
}

void FirstTimeDialog::slotSetup()
{
    if (confPageProxyOptions->localOption->isChecked()) {
        if (!Controller::certificateAlreadyGenerated()) {
            delete m_controller;
            m_controller = new Controller(this);
            confPageTLSCertificate->plainTextEdit->clear();
            connect(m_controller, &Controller::generationDone, this, [this]() {
                confPageTLSCertificate->installButton->setVisible(true);
                confPageTLSCertificate->installButton->setEnabled(true);
                confPageTLSCertificate->label->setText(
                    i18nc("@info", "About to install certificate with fingerprint: %1 ", m_controller->rootFingerprint()));
                confPageTLSCertificate->label->setVisible(true);
                confPageTLSCertificate->continueButton->setVisible(false);
            });
            ui->stack->setCurrentIndex(ConfPageTLSCertificate);

            connect(m_controller, &Controller::result, this, [this](KJob *) {
                if (m_controller->error()) {
                    confPageTLSCertificate->plainTextEdit->appendPlainText(m_controller->errorText());
                    return;
                }
                confPageTLSCertificate->installButton->setVisible(false);
                confPageTLSCertificate->continueButton->setVisible(true);
                confPageTLSCertificate->label->setText(i18nc("@info", "Installed certificate with fingerprint: %1", m_controller->rootFingerprint()));

                confPageTLSCertificate->continueButton->setEnabled(true);

            });
            connect(m_controller, &Controller::debutOutput, this, &FirstTimeDialog::slotTlsDebutOutput);
            m_controller->start();
        } else {
            startLocalServer();
            startWebsocketClient();
            generateManifest();
        }
    } else {
        generateManifest();
    }
}

void FirstTimeDialog::startLocalServer()
{
    if (m_serverProcess.state() != QProcess::NotRunning) {
        return;
    }

    m_serverProcess.start(u"gpgol-server"_s);
}

void FirstTimeDialog::startWebsocketClient()
{
    const auto clientId = QUuid::createUuid().toString(QUuid::WithoutBraces);
    auto &websocketClient = WebsocketClient::self(QUrl(u"wss://"_s + serverDomain() + u"/websocket"_s), clientId);

    connect(&websocketClient, &WebsocketClient::stateChanged, this, &FirstTimeDialog::slotStateChanged);
    connect(&websocketClient, &WebsocketClient::stateChanged, &m_systemTrayIcon, &SystemTrayIcon::slotStateChanged);

    m_systemTrayIcon.slotStateChanged(websocketClient.stateDisplay());
    slotStateChanged(websocketClient.stateDisplay());
}

void FirstTimeDialog::slotTlsDebutOutput(const QString &output)
{
    confPageTLSCertificate->plainTextEdit->appendPlainText(output);
}

void FirstTimeDialog::generateManifest()
{
    QFile file(u":/gpgol-client/manifest.xml.in"_s);
    if (!file.open(QIODeviceBase::ReadOnly)) {
        Q_ASSERT(false);
        return;
    }

    ui->stack->setCurrentIndex(ManifestPage);

    QByteArray manifest = file.readAll();
    manifest.replace("%HOST%", serverDomain().toUtf8());
    manifest.replace("%VERSION%", GPGOLWEB_VERSION_STRING);

    const auto saveFilePath = QStandardPaths::writableLocation(QStandardPaths::AppLocalDataLocation) + u"/gpgol-web-manifest.xml"_s;

    QSaveFile saveFile(saveFilePath);
    if (!saveFile.open(QIODeviceBase::WriteOnly)) {
        Q_ASSERT(false);
        return;
    }
    saveFile.write(manifest);
    saveFile.commit();

    confPageInstallAddin->manifestPath->setText(QDir::toNativeSeparators(saveFilePath));
}

QString FirstTimeDialog::serverDomain() const
{
    return confPageProxyOptions->localOption->isChecked() ? u"localhost:5656"_s : confPageProxyOptions->remoteServer->text();
}
