// SPDX-FileCopyrightText: 2025 g10 code GmbH
// SPDX-FileContributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "ewsgetfoldercontentrequest.h"

#include "ews/ewsfinditemrequest.h"
#include "ews/ewsgetitemrequest.h"
#include "ews/ewsitemshape.h"

EwsGetFolderContentRequest::EwsGetFolderContentRequest(const EwsId &folderId, const EwsClient &client)
    : KCompositeJob()
    , m_folderId(folderId)
    , m_ewsClient(client)
{
}

EwsGetFolderContentRequest::~EwsGetFolderContentRequest()
{
    delete m_currentFindItemRequest;
}

void EwsGetFolderContentRequest::start()
{
    EwsItemShape itemShape(EwsShapeIdOnly);

    m_currentFindItemRequest = new EwsFindItemRequest(m_ewsClient, this);
    m_currentFindItemRequest->setAutoDelete(false);
    m_currentFindItemRequest->setFolderId(m_folderId);
    m_currentFindItemRequest->setItemShape(itemShape);
    m_currentFindItemRequest->setTraversal(EwsTraversalType::EwsTraversalShallow);
    m_currentFindItemRequest->setPagination(EwsBasePointBeginning, 0, 20);

    connect(m_currentFindItemRequest, &EwsFindItemRequest::finished, this, &EwsGetFolderContentRequest::emailListFetched);

    m_currentFindItemRequest->start();
}

void EwsGetFolderContentRequest::emailListFetched(KJob *job)
{
    auto findItemRequest = qobject_cast<EwsFindItemRequest *>(job);
    Q_ASSERT(findItemRequest);

    const auto items = findItemRequest->items();
    m_totalItems = findItemRequest->totalItems();
    m_offset = findItemRequest->nextOffset();

    // TODO I think we also need error handling, here. The assert below once triggered for me due to what I think was
    //      a stale ews access token. But also consider a network problem causing the n-th in a series of jobs to
    //      fail. We shoud pass on an error condition, in this case.
    Q_ASSERT(findItemRequest->nextOffset() <= findItemRequest->pageOffset() + 20);

    EwsId::List ids;
    for (const auto &item : items) {
        ids << item[EwsItemFieldItemId].value<EwsId>();
    }

    if (ids.isEmpty()) {
        // This actually happens, if folder size is a multiple of our fetch size (20). In that case
        // EwsFindItemRequest::includesLastItem() does not appear to be reliable. MS bug?
        qWarning() << "At end of the email list" << findItemRequest->nextOffset();
        emitResult();
        return;
    }

    EwsItemShape itemShape(EwsShapeIdOnly);
    itemShape.setFlags(EwsItemShape::IncludeMimeContent);

    auto getItemRequest = new EwsGetItemRequest(m_ewsClient, this);
    getItemRequest->setItemIds(ids);
    getItemRequest->setItemShape(itemShape);
    connect(getItemRequest, &EwsGetItemRequest::finished, this, &EwsGetFolderContentRequest::emailContentFetched);

    getItemRequest->start();
}

void EwsGetFolderContentRequest::emailContentFetched(KJob *job)
{
    auto getItemRequest = qobject_cast<EwsGetItemRequest *>(job);
    Q_ASSERT(getItemRequest);

    const auto responses = getItemRequest->responses();
    QList<EwsItem> items;
    for (const auto &response : responses) {
        items << response.item();
    }

    Q_EMIT fetchedItems(items);

    if (!m_currentFindItemRequest->includesLastItem()) {
        EwsItemShape itemShape(EwsShapeIdOnly);

        auto nextFindItemRequest = new EwsFindItemRequest(m_ewsClient, this);
        nextFindItemRequest->setAutoDelete(false);
        nextFindItemRequest->setFolderId(m_folderId);
        nextFindItemRequest->setItemShape(itemShape);
        nextFindItemRequest->setTraversal(EwsTraversalType::EwsTraversalShallow);
        nextFindItemRequest->setPagination(EwsBasePointBeginning, m_currentFindItemRequest->nextOffset(), 20);

        connect(nextFindItemRequest, &EwsFindItemRequest::finished, this, &EwsGetFolderContentRequest::emailListFetched);

        m_currentFindItemRequest->deleteLater();
        m_currentFindItemRequest = nextFindItemRequest;
        nextFindItemRequest->start();
        return;
    } else {
        m_currentFindItemRequest->deleteLater();
        m_currentFindItemRequest = nullptr;
        emitResult();
    }
}

int EwsGetFolderContentRequest::totalItems() const
{
    return m_totalItems;
}

int EwsGetFolderContentRequest::offset() const
{
    return m_offset;
}
